#include "libTextureControl.h"

#define DEBUG
#define HIGHLEVEL
#define PI (3.141592654)
#define TWOPI (2*PI)

#define	PIXELSIZE	150
#define	XPIXELSIZE	PIXELSIZE
#define	YPIXELSIZE	PIXELSIZE


#define APPROXCARRIERFREQUENCY 60000
#define	WAVESPERSAMPLE	6
#define SAMPLESPERPERIOD 170
#define LOOKAHEADSAMPLES 80
#define CROSSFADESAMPLES 20

#define	MAXTEXTURES	30
#define	MAXTAXTELS	50

#define FT5XADDR  0x38
#define FINGERTHRESHOLD 3

#define	GPIOPROOF	0	// debug output
#define	GPIOINT		2	// interrupt request input
#define STMRESET	5
#define	STMLOAD		4
#define	STM32SPI	0

#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/spi/spidev.h>
#include <linux/i2c-dev.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <unistd.h>
#include <math.h>
#include <signal.h>
#include <string.h>
#include <pthread.h>



void (*pressedCallback) (int,int) = NULL;
void (*draggedCallback) (int,int) = NULL;
void (*releasedCallback) () = NULL;
void (*speedCallback) (int,int) = NULL;
void (*textureCallback) (int,int) = NULL;


void registerCallbackPressed(void (*myfonction) (int x, int y)){

	pressedCallback = myfonction;
}

void registerCallbackDragged(void (*myfonction) (int x, int y)){

	draggedCallback = myfonction;
}

void registerCallbackRelease(void (*myfonction)()){

	releasedCallback = myfonction;
}

void registerCallbackSpeed(void (*myfonction) (int speedX, int speedY)){

	speedCallback = myfonction;
}

void registerCallbackTexture(void (*myfonction) (int signal, int period)){

	textureCallback = myfonction;
}


#ifdef HIGHLEVEL

int cbX,cbY;
int isPressed = 0;
int isDragged = 0;
int isReleased = 0;
int cbSpeedX, cbSpeedY, cbSignal, cbPeriod;

int getX(){
	return cbX;
}

int getY(){
	return cbY;
}
int getPressed(){
	int tmp = isPressed;
	isPressed = 0;
	return tmp;
}

int getDragged(){
	int tmp = isDragged;
	isDragged = 0;
	return tmp;
}

int getReleased(){
	int tmp = isReleased;
	isReleased = 0;
	return tmp;
}

int getSpeedX(){
	return cbSpeedX;
}

int getSpeedY(){
	return cbSpeedY;
}

int getPeriod(){
	return cbPeriod;
}

int getSignal(){
	return cbSignal;
}


void updatePressed(int x, int y){
	cbX = x;
	cbY = y;
	isPressed = 1;
}

void updateDragged(int x, int y){
	cbX = x;
	cbY = y;
	isDragged = 1;
}

void updateReleased(){
	isReleased = 1;
}

void updateSpeed(int speedX, int speedY){
	cbSpeedX = speedX;
	cbSpeedY = speedX;
}

void updateTexture(int s, int period){
	cbSignal = s;
	cbPeriod = period;
}

#endif



typedef struct {
} TypeCosineParam;

typedef struct {
	float alpha;	// alpha, 0..PI
} TypeRectParam;

typedef union {
	TypeCosineParam CosineP;
	TypeRectParam RectP;
} TypeTextureParam;

struct {
	float (*UsAngleSpeedFun)(int, float, float);
	int (*UsAmplFun)(int, float);
	float offs, ampl;	// US amplitude
	float sp;		// spatial period
	TypeTextureParam t;
} Textures[MAXTEXTURES];


// the finger position, given by fx & fy, is suppose to be localized in one of several shapes

// a rectangle is horizontal / vertical
typedef struct {
	int x1, y1, x2, y2;
} TypeRectCoord;

// an ellipse is given by 2 poles and a rope length
typedef struct {
	int x1, y1, x2, y2;
	int r;
} TypeEllipseCoord;

// for now, only 2 kinds of shape are defined, but we should add at least a polygon shape
typedef union {
	TypeRectCoord rc;
	TypeEllipseCoord ec;
} TypeShapeCoord;

// a taxtel is a shape, with a set of coordinate and a periodic texture in it
struct {
	int (*IsInsideFun)(int, int, int);
	TypeShapeCoord s;
	int t;
	int enabled;
} Taxtels[MAXTAXTELS];





pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

char i2cdevname[] = "/dev/i2c-3";
int i2cfd;
unsigned char i2cbuf[64];
int fp, fs, fi, fx, fy;
int fvx, fvy;
volatile unsigned int ISRCounter = 0;

unsigned char amplitudes[2 + 2*(SAMPLESPERPERIOD+CROSSFADESAMPLES)]; //provision for 1 additional command / period and crossfade with next period



float LineicASpeed(int tex, float vx, float vy) {
	float v = sqrt(vx*vx + vy*vy);		// speed in micrometers/period
	return v * TWOPI / (SAMPLESPERPERIOD * Textures[tex].sp);
}

float HorizASpeed(int tex, float vx, float vy) {
	float v = abs(vx);			// speed in micrometers/period
	return v * TWOPI / (SAMPLESPERPERIOD * Textures[tex].sp);
}

float VertASpeed(int tex, float vx, float vy) {
	float v = abs(vy);			// speed in micrometers/period
	return v * TWOPI / (SAMPLESPERPERIOD * Textures[tex].sp);
}

int UsOff(int tex, float a) {
	return 0;
}

int UsCosine(int tex, float a) {
	return Textures[tex].offs + Textures[tex].ampl * cos(a);
}

int UsRect(int tex, float a) {
	float r;
	float f = Textures[tex].t.RectP.alpha;
	float v = 0.5;
	for (r = 1; r < 20; ++r) v += (sin(f*r)/(f*r)) * cos(a*r);
	return Textures[tex].offs + Textures[tex].ampl * (v*4.0*(f/PI)-1.0);
}

//*********************************** 
//*********************************** 
//**** Taxtel Shape Functions  ****** 
//*********************************** 
//*********************************** 
int IsOutside(int T, int x, int y) {
	return 0;
}

int IsInRect(int T, int x, int y) {
	return (x >= Taxtels[T].s.rc.x1)
	&& (x < Taxtels[T].s.rc.x2)
	&& (y >= Taxtels[T].s.rc.y1)
	&& (y < Taxtels[T].s.rc.y2);
}

int IsInEllipse(int T, int x, int y) {
	float dx1, dy1, dx2, dy2;
	float d1, d2;
	dx1 = x - Taxtels[T].s.ec.x1; dy1 = y - Taxtels[T].s.ec.y1;
	dx2 = x - Taxtels[T].s.ec.x2; dy2 = y - Taxtels[T].s.ec.y2;
	d1 = sqrt(dx1*dx1 + dy1*dy1);
	d2 = sqrt(dx2*dx2 + dy2*dy2);
	return (d1 + d2) <= Taxtels[T].s.ec.r;
}

//*********************************** 
//*********************************** 
//********** READ FINGER DATA  ****** 
//*********************************** 
//*********************************** 

int ReadFinger() {
	i2cbuf[0] = 0;
	if (write(i2cfd, i2cbuf, 1) != 1) {
		fprintf(stderr, "ERROR HANDLING: i2c read address failed !\n");
		return -1;
	}
	if (read(i2cfd, i2cbuf, 7) != 7) {
		fprintf(stderr, "ERROR HANDLING: i2c read data failed !\n");
		return -1;
	}
    	unsigned int n = i2cbuf[2];
	if (0 != n) {
		unsigned int b, r=3;
		b = i2cbuf[r++]; fs = b >> 6; fx = (b & 0x0F) << 8;
		b = i2cbuf[r++]; fx |= b;
		b = i2cbuf[r++]; fi = b >> 4; fy = (b & 0x0F) << 8;
		b = i2cbuf[r]; fy |= b;
		fp = 1;
		fx *= XPIXELSIZE; // micrometers/pixel
		fy *= YPIXELSIZE; // micrometers/pixel
	} else fp = 0;
	return 0;
}

//****************************************** 
//****************************************** 
//********** SPI WRITE & READ U16 ********** 
//****************************************** 
//****************************************** 
//data : Word to write.  Content is overwritten with Word read. 



int SpiWriteAndReadU16 (int spi_device, unsigned short int *U16) {
	unsigned char x[2];


	x[1] = *U16 & 0x00FF; x[0] = *U16 >> 8;
	wiringPiSPIDataRW(spi_device, x, 2);
	*U16 = (x[0] << 8) | x[1];

	return 0;
}

void nexta(float *a, float da){
	*a += da;
	if (*a > TWOPI) *a -= TWOPI; else
	if (*a < 0.0) *a += TWOPI;
}


//////////////////////////////
// Setting TaxtTels and Textures
//////////////////////////////




void setTaxtTelRect(int id,int textureId, int x1, int y1, int x2, int y2){
	if (id<MAXTAXTELS){
		pthread_mutex_lock(&mutex);
		Taxtels[id].s.rc.x1 = x1 * XPIXELSIZE;
		Taxtels[id].s.rc.y1 = y1 * YPIXELSIZE;
		Taxtels[id].s.rc.x2 = x2 * XPIXELSIZE;
		Taxtels[id].s.rc.y2 = y2 * YPIXELSIZE;
		Taxtels[id].IsInsideFun = IsInRect;
		Taxtels[id].t = textureId;
		Taxtels[id].enabled = 1;
		pthread_mutex_unlock(&mutex);
#ifdef DEBUG
		printf("TAXTEL: %i, Type: rect, text= %i, x1 = %i, y1 = %i, x2 = %i, y2 = %i\n", id, textureId, x1, y1, x2, y2);
	
#endif
	}
}

void setTaxtTelEllipse(int id,int textureId, int x1, int y1, int x2, int y2, int r){
	if (id<MAXTAXTELS){
		pthread_mutex_lock(&mutex);
		Taxtels[id].s.rc.x1 = x1 * XPIXELSIZE;
		Taxtels[id].s.rc.y1 = y1 * YPIXELSIZE;
		Taxtels[id].s.rc.x2 = x2 * XPIXELSIZE;
		Taxtels[id].s.rc.y2 = y2 * YPIXELSIZE;
		Taxtels[id].s.ec.r =  r * PIXELSIZE;
		Taxtels[id].IsInsideFun = IsInRect;
		Taxtels[id].t = textureId;
		Taxtels[id].enabled = 1;
		pthread_mutex_unlock(&mutex);
#ifdef DEBUG
		printf("TAXTEL: %i, Type: ellipse, text= %i, x1 = %i, y1 = %i, x2 = %i, y2 = %i, r = %i\n", id, textureId, x1, y1, x2, y2 , r);
	
#endif
	}
}




void disableTaxTel(int id){
	if (id<MAXTAXTELS){
		pthread_mutex_lock(&mutex);
		//Taxtels[id].IsInsideFun = IsOutside;
		//Taxtels[id].t = textureId;
		Taxtels[id].enabled = 0;
		pthread_mutex_unlock(&mutex);
#ifdef DEBUG
		printf("Disable taxtTel: %i\n", id);
	
#endif
	}
}

void enableTaxTel(int id){
	if(id<MAXTAXTELS){
		Taxtels[id].enabled =1;
		#ifdef DEBUG
		printf("Enable taxtTel: %i\n", id);
	
		#endif
	}
}

void setTextureRect(int textureId,float offset,float amplitude,float period,float ratio,char* speedFunc){
#ifdef DEBUG
	printf("Texture: %i, Type: rect, offset= %f, amplitude = %f, period = %f, ratio = %f\n", textureId, offset, amplitude, period, ratio);
#endif
	pthread_mutex_lock(&mutex);
	if (textureId<MAXTEXTURES){
		Textures[textureId].UsAmplFun = UsRect;
		Textures[textureId].offs = offset;
		Textures[textureId].ampl = amplitude;
		Textures[textureId].sp = period;
		Textures[textureId].t.RectP.alpha = ratio;
		if(strcmp(speedFunc,"vertical") == 0)
			Textures[textureId].UsAngleSpeedFun = VertASpeed;
		else
			if (strcmp(speedFunc, "horizontal") == 0)
				Textures[textureId].UsAngleSpeedFun = HorizASpeed;
			else
				Textures[textureId].UsAngleSpeedFun = LineicASpeed;
	}
	pthread_mutex_unlock(&mutex);
}

void setTextureCos(int textureId,float offset,float amplitude,float period, char* speedFunc){
#ifdef DEBUG
	printf("Texture: %i, Type: cos, offset= %f, amplitude = %f, period = %f\n", textureId, offset, amplitude, period);
#endif
	pthread_mutex_lock(&mutex);
	if (textureId<MAXTEXTURES){
		Textures[textureId].UsAmplFun = UsCosine;
		Textures[textureId].offs = offset;
		Textures[textureId].ampl = amplitude;
		Textures[textureId].sp = period;
		if (strcmp(speedFunc, "vertical") == 0)
			Textures[textureId].UsAngleSpeedFun = VertASpeed;
		else
			if (strcmp(speedFunc, "horizontal") == 0)
				Textures[textureId].UsAngleSpeedFun = HorizASpeed;
			else
				Textures[textureId].UsAngleSpeedFun = LineicASpeed;
	}
	pthread_mutex_unlock(&mutex);
}




void stopEvita(){

	digitalWrite(STMRESET, LOW);
	delay(200);
	digitalWrite(STMRESET, HIGH);
	delay(200);

	close(i2cfd);
}



void MyISR(void) {			// this interrupt MUST execute once every period (18ms)
	unsigned short int b16;
	unsigned int n;
	int ofp, ofx, ofy;
	int tax, tex;
	static float a = 0.0, da = 0.233;

	digitalWrite(GPIOPROOF, HIGH);	// rising slope signals start of idle time

	ofp = fp; ofx = fx; ofy = fy;
	ReadFinger();


	if (!ofp) {
		ofx = fx; ofy = fy;
		fvx = fvy = 2000;
	}
	fvx = (2 * fvx + fx - ofx) / 3; fvy = (2 * fvy + fy - ofy) / 3;	// speeds in micrometers/period

	if(1/*sendSpeed*/) speedCallback(fvx,fvy);

	pthread_mutex_lock(&mutex);

	for (tax = MAXTAXTELS-1; tax > 0; tax--) {	
													// 0 is unconditionnal background
		if (Taxtels[tax].enabled && Taxtels[tax].IsInsideFun(tax, fx, fy)) break;

	}
	//printf("%d\n", tax);
	tex = Taxtels[tax].t;
	if (1/*sendTexture*/) textureCallback(Textures[tex].UsAmplFun == UsRect, Textures[tex].sp);

#ifdef DEBUGPRINTF
	if (fp != ofp) fprintf(stdout, "\n");
	fprintf(stdout, "%1d %7d %7d %7d %7d %2d\r", fp, fx, fy, fvx, ISRCounter, tex);
	fflush(stdout); 
#endif

// adjust next period parameters according to finger position/speed

// compute texture and send it to stm32 for next period
	amplitudes[0] = amplitudes[1] = 0; // use this location for extra payload as needed 
	for (n = 2; n < 2 + 2*(SAMPLESPERPERIOD+CROSSFADESAMPLES); n += 2) amplitudes[n] = 0x01;
	if (ofp || fp) {
		float aa;
		int k = 0;
		for (n = 3; n < 2 + 2*(CROSSFADESAMPLES); n += 2) {
			amplitudes[n] = amplitudes[n+2*SAMPLESPERPERIOD] + (k++ * Textures[tex].UsAmplFun(tex, a) + CROSSFADESAMPLES/2)/CROSSFADESAMPLES;
			nexta(&a, da);
		}
		for (; n < 2 + 2*(SAMPLESPERPERIOD); n += 2) {
			amplitudes[n] = Textures[tex].UsAmplFun(tex, a);
			nexta(&a, da);
		}
		aa = a;
		for (; n < 2 + 2*(SAMPLESPERPERIOD+CROSSFADESAMPLES); n += 2) {
			amplitudes[n] = (--k * Textures[tex].UsAmplFun(tex, aa) + CROSSFADESAMPLES/2)/CROSSFADESAMPLES;
			nexta(&aa, da);
		}


	} else for (n = 3; n < 2 + 2*(SAMPLESPERPERIOD+CROSSFADESAMPLES); n += 2) amplitudes[n] = 0;

	// Dever Mutex
	pthread_mutex_unlock(&mutex);

	wiringPiSPIDataRW(STM32SPI, amplitudes, 2 + 2*SAMPLESPERPERIOD);
	//printf("MYISR\n");
	if (1/*sendPosition*/) {
		if (!ofp) {
			if (fp)	/*lo_send(sendTo, "/pressed", "ii", fx/XPIXELSIZE, fy/YPIXELSIZE);if(pressedCallback != NULL) */ pressedCallback(fx / XPIXELSIZE, fy / YPIXELSIZE);
			//printf("X: %d, Y: %d\n", getX(), getY());
		}
		else {
			if (fp)	/*lo_send(sendTo, "/dragged", "ii", fx/XPIXELSIZE, fy/YPIXELSIZE);*/draggedCallback(fx / XPIXELSIZE, fy / YPIXELSIZE);
			else /*lo_send(sendTo, "/released", "");*/ releasedCallback();
		}
	}
	digitalWrite(GPIOPROOF, LOW);	// falling slope signals start of new period
}



int startEvita(){
printf("Debut start \n");

if (*pressedCallback == NULL) registerCallbackPressed(updatePressed);
if (*draggedCallback == NULL) registerCallbackDragged(updateDragged);
if (*releasedCallback == NULL) registerCallbackRelease(updateReleased);
if (*speedCallback  == NULL) registerCallbackSpeed(updateSpeed);
if (*textureCallback == NULL) registerCallbackTexture(updateTexture);


	if (wiringPiSetup()) {
		fprintf(stderr, "wiringPiSetup failed!");
		return -1;
	}

	if (wiringPiISR(GPIOINT, INT_EDGE_FALLING, &MyISR)) {
		fprintf(stderr, "wiringPiISR failed!");
		return -1;
	}



	//TextureControl::SpiWriteAndReadU16
	wiringPiSPISetup(0, 2000000); 	// 8 µs/sample is fast enough for cerb40 @ 60kHz plate

	pinMode(GPIOPROOF, OUTPUT);	// GPIO output for debugging purpose

	pinMode(STMRESET, OUTPUT);	// put slave in run mode, not firmware upload mode:
	pinMode(STMLOAD, OUTPUT);
	digitalWrite(STMLOAD, LOW);
	digitalWrite(STMRESET, LOW);
	delay(400);
	digitalWrite(STMRESET, HIGH);
	delay(300);


	int tax, tex;
	pthread_mutex_lock(&mutex);
	printf("Init tex tax\n");
	for (tex = MAXTEXTURES-1; tex >= 0; tex--) {
		Textures[tex].UsAngleSpeedFun = LineicASpeed;
		Textures[tex].UsAmplFun = UsOff;
	}
	printf("Start-> text:%d\n", tex);

	for (tax = MAXTAXTELS-1; tax >= 0; tax--) {
		Taxtels[tax].IsInsideFun = IsOutside;
		Taxtels[tax].t = 0;
		Taxtels[tax].enabled = 0;
	}

	printf("Start-> tax: %d\n", tax);
	pthread_mutex_unlock(&mutex);
	i2cfd = open(i2cdevname, O_RDWR);
	if (i2cfd < 0) {
		fprintf (stderr, "Unable to open %s for R/W\n", i2cdevname);
		return -1;
	}
	printf("I2C open\n");
	if (ioctl(i2cfd, I2C_SLAVE, FT5XADDR) < 0) {
		fprintf (stderr, "Unable to open device 0x%02X on %s for R/W\n", FT5XADDR, i2cdevname);
		return -1;
	}
	printf("ioctl\n");
	i2cbuf[0] = 0x80; i2cbuf[1] = FINGERTHRESHOLD;
	if (write(i2cfd, i2cbuf, 2) != 2) {
		fprintf(stderr, "Changing FT5X sensitivity failed !\n");
		return -1;
	}
printf("i2cbuf\n");
	int K;
	char inputBuffer[100];
	strcpy(inputBuffer, "");

sleep(1);
printf("Set texture\n");
setTextureCos(0,16,0,1000,"horizontal");
setTaxtTelRect(0,0, 0, 0, 500, 600);
printf("texture Set\n");

printf("Fin de start\n");

return 0;
	
}
/*
int sortie = 0;


int main(void)
{
	int first = 1;
	int second = 2;

	startEvita();


	setTextureRect(first,60,50,800,0.3,"vertical");
	printf("First Texture done\n");
	setTextureCos(second,70,100,500, "vertical");
	printf("Second Texture Done\n");

	setTaxtTelRect(first,first, 0, 0, 500, 600);
	setTaxtTelRect(second,second, 500, 0, 1000, 600);
	//setTaxtTelEllipse(second,second, 256*XPIXELSIZE,300*YPIXELSIZE,768*XPIXELSIZE,300*YPIXELSIZE,600*PIXELSIZE);
	printf("Taxtel Done\n");
		

	do {


	}while(!sortie);


	stopEvita();

	
	return 0;
}
*/