
'''
Support for the TelosB / Tmote Sky platforms, and close clones.

Utilizes the GoodFET firmware with CCSPI application, and the GoodFET client code.
'''
import os
import time
import struct
import time
from datetime import datetime, timedelta
from kbutils import KBCapabilities, makeFCS
from GoodFETCCSPI import GoodFETCCSPI

CC2420_REG_SYNC = 0x14

class TELOSB:
    def __init__(self, dev):
        '''
        Instantiates the KillerBee class for our TelosB/TmoteSky running GoodFET firmware.
        @type dev:   String
        @param dev:  Serial device identifier (ex /dev/ttyUSB0)
        @return: None
        @rtype: None
        '''
        self._channel = None
        self.handle = None
        self.dev = dev

        os.environ["board"] = "telosb" #set enviroment variable for GoodFET code to use
        self.handle = GoodFETCCSPI()
        self.handle.serInit(port=self.dev)
        self.handle.setup()

        self.__stream_open = False
        self.capabilities = KBCapabilities()
        self.__set_capabilities()

    def close(self):
        self.handle.serClose()
        self.handle = None

    def check_capability(self, capab):
        return self.capabilities.check(capab)
    def get_capabilities(self):
        return self.capabilities.getlist()
    def __set_capabilities(self):
        '''
        Sets the capability information appropriate for GoodFETCCSPI client and firmware.
        @rtype: None
        @return: None
        '''
        self.capabilities.setcapab(KBCapabilities.FREQ_2400, True)
        self.capabilities.setcapab(KBCapabilities.SNIFF, True)
        self.capabilities.setcapab(KBCapabilities.SETCHAN, True)
        self.capabilities.setcapab(KBCapabilities.INJECT, True)
        self.capabilities.setcapab(KBCapabilities.PHYJAM_REFLEX, True)
        self.capabilities.setcapab(KBCapabilities.SET_SYNC, True)
        return
    def get_rssi(self):
        return  self.handle.RF_getrssi()


    # KillerBee expects the driver to implement this function
    #def get_dev_info(self, dev, bus):
    def get_dev_info(self):
        '''
        Returns device information in a list identifying the device.
        @rtype: List
        @return: List of 3 strings identifying device.
        '''
        return [self.dev, "TelosB/Tmote", ""]

    # KillerBee expects the driver to implement this function
    def sniffer_on(self, channel=None):
        '''
        Turns the sniffer on such that pnext() will start returning observed
        data.  Will set the command mode to Air Capture if it is not already
        set.
        @type channel: Integer
        @param channel: Sets the channel, optional
        @rtype: None
        '''
        self.capabilities.require(KBCapabilities.SNIFF)

        self.handle.RF_promiscuity(1);
        self.handle.RF_autocrc(0);

        if channel != None:
            self.set_channel(channel)

        self.handle.CC_RFST_RX();
        print "Sniffer started (listening as %010x on %i MHz)" % (self.handle.RF_getsmac(), self.handle.RF_getfreq()/10**6);

        self.__stream_open = True

    # KillerBee expects the driver to implement this function
    def sniffer_off(self):
        '''
        Turns the sniffer off, freeing the hardware for other functions.  It is
        not necessary to call this function before closing the interface with
        close().
        @rtype: None
        '''
        #TODO actually have firmware stop sending us packets!
        self.__stream_open = False

    # KillerBee expects the driver to implement this function
    def set_channel(self, channel):
        '''
        Sets the radio interface to the specifid channel (limited to 2.4 GHz channels 11-26)
        @type channel: Integer
        @param channel: Sets the channel, optional
        @rtype: None
        '''
        self.capabilities.require(KBCapabilities.SETCHAN)

        if channel >= 11 or channel <= 26:
            self._channel = channel
            self.handle.RF_setchan(channel)
        else:
            raise Exception('Invalid channel')
    def get_channel(self):
        return self._channel
    def get_handle(self):
        return self.handle

    # KillerBee expects the driver to implement this function
    def inject(self, packet, channel=None, count=1, delay=0):
        '''
        Injects the specified packet contents.
        @type packet: String
        @param packet: Packet contents to transmit, without FCS.
        @type channel: Integer
        @param channel: Sets the channel, optional
        @type count: Integer
        @param count: Transmits a specified number of frames, def=1
        @type delay: Float
        @param delay: Delay between each frame, def=1
        @rtype: None
        '''
        self.capabilities.require(KBCapabilities.INJECT)

        if len(packet) < 1:
            raise Exception('Empty packet')
        if len(packet) > 125:                   # 127 - 2 to accommodate FCS
            raise Exception('Packet too long')

        if channel != None:
            self.set_channel(channel)

        self.handle.RF_autocrc(1)               #let radio add the CRC
        for pnum in range(0, count):
            gfready = [ord(x) for x in packet]  #convert packet string to GoodFET expected integer format
            gfready.insert(0, len(gfready)+2)   #add a length that leaves room for CRC
            self.handle.RF_txpacket(gfready)
            # Sleep was for 1 second but testing by Gianfranco Costamagna suggested lowering to 1/100th of a second
            time.sleep(0.01)                    #TODO get rid of completely, and just check CC2420 status
            # https://github.com/alvarop/msp430-cc2500/blob/master/lib/cc2500/cc2500.c

    def get_package(self,timeout=0.1):
        packet = None;
        start = time.time()
        while (packet is None and (time.time() - start < timeout)):
            packet = self.handle.RF_rxpacket()
        return packet

    # KillerBee expects the driver to implement this function
    def pnext(self, timeout=1000):
        '''
        Returns a dictionary containing packet data, else None.
        @type timeout: Integer
        @param timeout: Timeout to wait for packet reception in usec
        @rtype: List
        @return: Returns None is timeout expires and no packet received.  When a packet is received, a dictionary is returned with the keys bytes (string of packet bytes), validcrc (boolean if a vaid CRC), rssi (unscaled RSSI), and location (may be set to None). For backwards compatibility, keys for 0,1,2 are provided such that it can be treated as if a list is returned, in the form [ String: packet contents | Bool: Valid CRC | Int: Unscaled RSSI ]
        '''
        if self.__stream_open == False:
            self.sniffer_on() #start sniffing

        packet = None;
        start = datetime.utcnow()

        while (packet is None and (start + timedelta(microseconds=timeout) > datetime.utcnow())):
            packet = self.handle.RF_rxpacket()
            rssi = self.handle.RF_getrssi() #TODO calibrate

        if packet is None:
            return None

        frame = packet[1:]
        if frame[-2:] == makeFCS(frame[:-2]): validcrc = True
        else: validcrc = False
        #Return in a nicer dictionary format, so we don't have to reference by number indicies.
        #Note that 0,1,2 indicies inserted twice for backwards compatibility.
        result = {0:frame, 1:validcrc, 2:rssi, 'bytes':frame, 'validcrc':validcrc, 'rssi':rssi, 'location':None,'packet':packet}
        result['dbm'] = rssi - 45 #TODO tune specifically to the Tmote platform (does ext antenna need to different?)
        result['datetime'] = datetime.utcnow()
        return result
    def test_rssi(self, timeout=1000):
        '''
        Returns a dictionary containing packet data, else None.
        @type timeout: Integer
        @param timeout: Timeout to wait for packet reception in usec
        @rtype: List
        @return: Returns None is timeout expires and no packet received.  When a packet is received, a dictionary is returned with the keys bytes (string of packet bytes), validcrc (boolean if a vaid CRC), rssi (unscaled RSSI), and location (may be set to None). For backwards compatibility, keys for 0,1,2 are provided such that it can be treated as if a list is returned, in the form [ String: packet contents | Bool: Valid CRC | Int: Unscaled RSSI ]
        '''
        if self.__stream_open == False:
            self.sniffer_on() #start sniffing

        packet = None;
        start = datetime.utcnow()

        while (packet is None and (start + timedelta(microseconds=timeout) > datetime.utcnow())):
            packet = self.handle.RF_rxpacket()
            rssi = self.handle.RF_getrssi() #TODO calibrate
        return rssi
    def ping(self, da, panid, sa, channel=None):
        '''
        Not yet implemented.
        @return: None
        @rtype: None
        '''
        raise Exception('Not yet implemented')

    def jammer_on(self, channel=None):
        '''
        Not yet implemented.
        @type channel: Integer
        @param channel: Sets the channel, optional
        @rtype: None
        '''
        self.capabilities.require(KBCapabilities.PHYJAM_REFLEX)

        self.handle.RF_promiscuity(1)
        self.handle.RF_autocrc(0)
        if channel != None:
            self.set_channel(channel)
        self.handle.CC_RFST_RX()
        self.handle.RF_reflexjam()

    def set_sync(self, sync=0xA70F):
        '''Set the register controlling the 802.15.4 PHY sync byte.'''
        self.capabilities.require(KBCapabilities.SET_SYNC)
        if (sync >> 16) > 0:
            raise Exception("Sync word (%x) must be 2-bytes or less." % sync)
        return self.handle.poke(CC2420_REG_SYNC, sync)

    def jammer_off(self, channel=None):
        '''
        Not yet implemented.
        @return: None
        @rtype: None
        '''
        #TODO implement
        raise Exception('Not yet implemented')


test_rssi.py(messurer rssi):

# -*- coding: utf-8 -*-

from dev_telosb import TELOSB
import time

count = 0
telosb = TELOSB("/dev/ttyUSB0")
telosb.set_channel(0x19)
telosb.sniffer_on()
start = time.time()
for i in range(100000):
    print(telosb.test_rssi())
    while(time.time() - start < 1):
        pass
    start = time.time()
    #mes = telosb.pnext(1000)
    #if(mes != None):
    #    print(mes["dbm"])
    #else:
    #    print(telosb.get_handle().RF_getrssi())

telosb.close()


sniffer_test.py(sniffer les message):

# -*- coding: utf-8 -*-

from dev_telosb import TELOSB
from dev_apimote import APIMOTE
from zigbeedecode import ZigBeeNWKPacketParser
from zigbeedecode import ZigBeeAPSPacketParser

import array

<nowiki>
####################################################
#报文说明
#0x17   -> length (总字节数 - 1)
#0x61   -> 报头引导
#0x88   -> 报头引导
#0xd    -> frameID(不确定, 它随报文顺序,而增加)
#0x34   -> PID(LOW)
#0x12   -> PID(HIGH)
#0x3    -> Addr16 of sender (LOW)
#0x0    -> Addr16 of sender (HIGH)
#0x1    -> Addr16 of receiver (LOW)
#0x0    -> Addr16 of receiver (HIGH)
#0xeb   ->  计数器?
#0x0    ->  计数器?
#0x68   -
#0x65   -
#0x6c   -
#0x6c   -
#0x6f   -       >    data
#0x20   -
#0x77   -
#0x6f   -
#0x72   -
#0x6c   -
#0xe7   ->  checkSum(LOW)
#0xa9   ->  checkSum(HIGH)
####################################################

</nowiki>

DEVICE = "/dev/ttyUSB0"

class parser_trame:
    def __init__(self, packet=None):
        if packet is None:
            self.mes = None
            return
        mes = array.array('B', packet['packet'])
        self.mes = mes
    def get_message(self, mes):
        mes = array.array('B', mes['packet'])
        self.mes = mes
    def get_len(self):
        if self.mes is not None:
            return self.mes[0]
    def get_frameID(self):
        if self.mes is not None:
            return self.mes[3]
    def get_PINID_LOW(self):
        if self.mes is not None:
            return self.mes[4]
    def get_PINID_HIGH(self):
        if self.mes is not None:
            return self.mes[5]
    def get_addr16_sender_low(self):
        if self.mes is not None:
            return self.mes[6]
    def get_addr16_sender_high(self):
        if self.mes is not None:
            return self.mes[7]
    def get_addr16_receiver_low(self):
        if self.mes is not None:
            return self.mes[8]
    def get_addr16_receiver_high(self):
        if self.mes is not None:
            return self.mes[9]
    def get_data(self):
        if self.mes is not None:
            return self.mes[12:-2]


def main():
    telosb = TELOSB(DEVICE)
    telosb.sniffer_on(0x19)
    nwk = ZigBeeNWKPacketParser()
    aps = ZigBeeAPSPacketParser()
    parser = parser_trame()
    while(True):
        mes = telosb.pnext()
        if(mes != None):
            print('----------------------------------------')
            parser.get_message(mes)
            print(parser.get_data())
            for i in parser.get_data():
                print('%#x'%i)
                print('%c'%i)

if __name__ == "__main__":
    main()
